/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: OS_Win32.c $:

  Description:
    Windows OS abstraction for netXTransport toolkit

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2020-11-18  Added missing "void" in the OS_GetMilliSecCounter() function definition
    2013-02-23  initial version

**************************************************************************************/

#include "OS_Dependent.h"
#include <netXTransport_Errors.h>
#include <OS_Includes.h>

/*****************************************************************************/
/*! Create Lock (Usually same as mutex, but does not support timed waiting)
*     \return Handle to created lock                                         */
/*****************************************************************************/
void* OS_CreateLock(void)
{
  CRITICAL_SECTION* ptSection = (CRITICAL_SECTION*)OS_Memalloc(sizeof(CRITICAL_SECTION));

  if(NULL != ptSection)
  {
    InitializeCriticalSection(ptSection);
  }
  return ptSection;
}

/*****************************************************************************/
/*! Acquire a lock
*     \param pvLock Handle to lock                                           */
/*****************************************************************************/
void OS_EnterLock(void* pvLock)
{
  CRITICAL_SECTION* ptSection = (CRITICAL_SECTION*)pvLock;

  EnterCriticalSection(ptSection);
}

/*****************************************************************************/
/*! Release a lock
*     \param pvLock Handle to lock                                           */
/*****************************************************************************/
void OS_LeaveLock(void* pvLock)
{
  CRITICAL_SECTION* ptSection = (CRITICAL_SECTION*)pvLock;

  LeaveCriticalSection(ptSection);
}

/*****************************************************************************/
/*! Delete a lock
*     \param pvLock Handle to lock                                           */
/*****************************************************************************/
void OS_DeleteLock(void* pvLock)
{
  CRITICAL_SECTION* ptSection = (CRITICAL_SECTION*)pvLock;

  DeleteCriticalSection(ptSection);
  free(ptSection);
}

/*****************************************************************************/
/*! Writes formatted string to szDest
*     \param szDest   Pointer to destiantion string
*     \param ulSize   Size of destination string
*     \param format   Pointer to format string
*     \return Pointer to szDest                                              */
/*****************************************************************************/
char* OS_Strvsprintf(char *szDest, uint32_t ulSize, const char *format, ...)
{
  int     iRet;
  va_list ap;

  va_start(ap, format);
  iRet = vsprintf_s( szDest, ulSize, format, ap);
  va_end(ap);

  return szDest;
}

/*****************************************************************************/
/*! Compare strings
*     \param pszBuf1  String buffer 1
*     \param pszBuf2  String buffer 2
*     \return 0 if strings are equal                                         */
/*****************************************************************************/
int OS_Strcmp(const char* pszBuf1, const char* pszBuf2)
{
  return strcmp(pszBuf1, pszBuf2);
}

/*****************************************************************************/
/*! Compare strings (case sensitive)
*     \param pszBuf1  String buffer 1
*     \param pszBuf2  String buffer 2
*     \return 0 if strings are equal                                         */
/*****************************************************************************/
int OS_Strnicmp (const char* pszBuf1, const char* pszBuf2, uint32_t len)
{
  return _strnicmp(pszBuf1, pszBuf2, len);
}

/*****************************************************************************/
/*! Copy string to destination buffer
*     \param szDest   Destination string
*     \param szSource Source string
*     \param ulLen    Maximum length to copy
*     \return Pointer to szDest                                              */
/*****************************************************************************/
char* OS_Strncpy(char* szDest, const char* szSource, uint32_t ulLen)
{
  strncpy(szDest, szSource, ulLen);

  return szDest;
}

/*****************************************************************************/
/*! Get length of string
*     \param szText  Text buffer
*     \return Length of given string                                         */
/*****************************************************************************/
int OS_Strlen(const char* szText)
{
  return (int)strlen(szText);
}

/*****************************************************************************/
/*! Find string occurrence
*     \param szText  Text buffer
*     \return Length of given string                                         */
/*****************************************************************************/
uint32_t OS_Strcspn( char* szSearchStr, char* szFindStr)
{
  return (uint32_t)strcspn( szSearchStr, szFindStr);
}

/*****************************************************************************/
/*! Concatenate strings
*     \param szDest    Pointer to destiantion string
*     \param szSrc     Pointer to source string
*     \param ulDstSize Maximum size of destiantion string
*     \return szDest                                                         */
/*****************************************************************************/
char* OS_Strncat(char* szDest, char* szSrc, uint32_t ulDstSize)
{
  strcat_s( szDest, ulDstSize, szSrc);

  return szDest;
}

/*****************************************************************************/
/*! Memory allocation
*     \param ulSize Size of block to allocate
*     \return NULL on failure                                                */
/*****************************************************************************/
void* OS_Memalloc(uint32_t ulSize)
{
  if ( 0 == ulSize)
    return NULL;

  return malloc(ulSize);
}

/*****************************************************************************/
/*! Memset
*     \param pvMem   Memory to set
*     \param bFill   Fill byte
*     \param ulSize  Size of the fill block                                  */
/*****************************************************************************/
void OS_Memset(void* pvMem, uint8_t bFill, uint32_t ulSize)
{
  memset(pvMem, bFill, ulSize);
}

/*****************************************************************************/
/*! Memcopy
*     \param pvDest  Destination pointer
*     \param pvSrc   Source pointer
*     \param ulSize  Size to copy                                            */
/*****************************************************************************/
void OS_Memcpy(void* pvBuf1, void* pvBuf2, uint32_t ulSize)
{
  memcpy(pvBuf1, pvBuf2, ulSize);
}

/*****************************************************************************/
/*! Memory resize
*     \param pvMem      Block to resize
*     \param ulNewSize  New size of the block
*     \return NULL on error                                                  */
/*****************************************************************************/
void* OS_Memrealloc(void* pvBuffer, uint32_t ulSize)
{
  return realloc(pvBuffer,ulSize);
}

/*****************************************************************************/
/*! Memcompare wrapper
*     \param pvBuf1  First compare buffer
*     \param pvBuf2  Second compare buffer
*     \param ulSize  Size to compare
*     \return 0 if blocks are equal                                          */
/*****************************************************************************/
int OS_Memcmp(void* pvBuf1, void* pvBuf2, uint32_t ulSize)
{
  return memcmp(pvBuf1, pvBuf2, ulSize);
}

/*****************************************************************************/
/*! Memory de-allocation
*     \param pvMem  Block to free                                            */
/*****************************************************************************/
void OS_Memfree(void* pvMem)
{
  free(pvMem);
}

/*****************************************************************************/
/*! Create event
*     \return Handle to created event                                        */
/*****************************************************************************/
void* OS_CreateEvent(void)
{
  return CreateEvent(NULL, FALSE, FALSE, NULL);
}

/*****************************************************************************/
/*! Signal event
*     \param pvEvent Handle to event                                         */
/*****************************************************************************/
void OS_SetEvent(void* pvEvent)
{
  SetEvent(pvEvent);
}

/*****************************************************************************/
/*! Reset event
*     \param pvEvent Handle to event                                         */
/*****************************************************************************/
void OS_ResetEvent(void* pvEvent)
{
  ResetEvent(pvEvent);
}

/*****************************************************************************/
/*! Delete event
*     \param pvEvent Handle to event                                         */
/*****************************************************************************/
void OS_DeleteEvent(void* pvEvent)
{
  CloseHandle(pvEvent);
}

/*****************************************************************************/
/*! Wait for event
*     \param pvEvent   Handle to event
*     \param ulTimeout Timeout in ms to wait for event
*     \return CIFX_EVENT_SIGNALLED if event was set, CIFX_EVENT_TIMEOUT otherwise */
/*****************************************************************************/
uint32_t OS_WaitEvent(void* pvEvent, uint32_t ulTimeout)
{
  uint32_t ulRet = OS_EVENT_TIMEOUT;

  if(WAIT_OBJECT_0 == WaitForSingleObject(pvEvent, ulTimeout))
    ulRet = OS_EVENT_SIGNALLED;

  return ulRet;
}

/*****************************************************************************/
/*! Create Semaphore
*     \param ulInitialCount
*     \return Pointer to semaphore object                                    */
/*****************************************************************************/
void* OS_CreateSemaphore(uint32_t ulInitialCount)
{
  return CreateSemaphore(NULL, ulInitialCount, MAXLONG, NULL);
}

/*****************************************************************************/
/*! Inc Semaphore
*     \param pvSem    Pointer to semaphore object
*     \param ulCount  value to increment                                     */
/*****************************************************************************/
void OS_PutSemaphore(void* pvSem, uint32_t ulCount)
{
  ReleaseSemaphore(pvSem, ulCount, NULL);
}

/*****************************************************************************/
/*! Deletes Semaphore
*     \param pvSem    Pointer to semaphore object                            */
/*****************************************************************************/
void OS_DeleteSemaphore(void* pvSem)
{
  CloseHandle(pvSem);
}

/*****************************************************************************/
/*! Wait for semaphore
*     \param pvSem   Pointer to semaphore object
*     \param ulTimeout Timeout in ms to wait for semaphore
*     \return OS_SEM_SIGNALLED if semaphore was set, OS_SEM_TIMEOUT otherwise */
/*****************************************************************************/
uint32_t OS_WaitSemaphore(void* pvSem, uint32_t ulTimeout)
{
  uint32_t ulRet = OS_SEM_TIMEOUT;

  if(WAIT_OBJECT_0 == WaitForSingleObject(pvSem, ulTimeout))
    ulRet = OS_SEM_SIGNALLED;

  return ulRet;
}

/*****************************************************************************/
/*! Returns currents system timer tick
*     \return timertick                                                      */
/*****************************************************************************/
uint32_t OS_GetMilliSecCounter( void)
{
  return GetTickCount();
}

/*****************************************************************************/
/*! Sleep
*     \param ulSleep Time to sleep in ms                                     */
/*****************************************************************************/
void OS_Sleep(uint32_t ulSleep)
{
  Sleep( ulSleep);
}

/*****************************************************************************/
/*! Creates thread
*     \param pvUserParam OS dependent thread parameter
*     \param ulParamSize size of buffer pointed by pvUserParam
*     \return NXT_NO_ERROR on success                                        */
/*****************************************************************************/
int32_t OS_CreateThread( void* pvUserParam, uint32_t ulParamSize)
{
  WIN32_THREAD_PARAM_T* ptWinThreadParam = (WIN32_THREAD_PARAM_T*)pvUserParam;
  int32_t               lRet             = NXT_INVALID_PARAMETER;

  if ((ptWinThreadParam == NULL) || (sizeof(WIN32_THREAD_PARAM_T) != ulParamSize))
    return lRet;

  ptWinThreadParam->hHandle = CreateThread( ptWinThreadParam->lptSecAtt,
                                            ptWinThreadParam->tStackSize,
                                            ptWinThreadParam->lpStartAddr,
                                            ptWinThreadParam->pvParam,
                                            ptWinThreadParam->dwFlags,
                                            &ptWinThreadParam->lpID);

  if (ptWinThreadParam->hHandle)
    lRet = NXT_NO_ERROR;

  return lRet;
}

/*****************************************************************************/
/*! Stops and delete thread resources
*     \param pvUserParam OS dependent thread parameter
*     \return NXT_NO_ERROR on success                                        */
/*****************************************************************************/
int32_t OS_DeleteThread( void* pvUserParam)
{
  WIN32_THREAD_PARAM_T* ptWinThreadParam = (WIN32_THREAD_PARAM_T*)pvUserParam;
  int32_t               lRet             = NXT_INVALID_PARAMETER;

  if (ptWinThreadParam == NULL)
    return lRet;

  WaitForSingleObject( ptWinThreadParam->hHandle,5000);

  return NXT_NO_ERROR;
}

/*****************************************************************************/
/*! Returns pointer to an interlocked variable
*     \return pointer to interlocked variable                                */
/*****************************************************************************/
void* OS_CreateInterLockedVariable(void)
{
  /* NOTE: There is no need to use the interlocked API if the used     */
  /*       variable is allocated per platform.                         */
  /*       - reads and writes to 32-bit variables (32-bit aligned) are */
  /*         atomic on 32-bit platform                                 */
  /*       - reads and writes to 64-bit variables (64-bit aligned) are */
  /*         atomic on 64-bit platform                                 */
#if _WIN32
  uint32_t volatile *plVar = (uint32_t volatile*)OS_Memalloc(sizeof(uint32_t volatile));
#else
  uint64_t volatile *plVar = (uint64_t volatile*)OS_Memalloc(sizeof(uint64_t volatile));
#endif
  if (NULL != plVar)
    *plVar = 0;

  return (void*)plVar;
}

/*****************************************************************************/
/*! Frees the interlocked variable, previously allocated by
    OS_CreateInterLockedVariable()
*     \param pvVal Pointer to the interlocked variable allocated by
                   OS_CreateInterLockedVariable()                            */
/*****************************************************************************/
void OS_DeleteInterLockedVariable(void* pvVal)
{
  OS_Memfree( pvVal);
}

/*****************************************************************************/
/*! Increments the current value in an atomic fashion
*     \param pvVal Pointer to the interlocked variable allocated by
                   OS_CreateInterLockedVariable()
*     \return current value+1                                                */
/*****************************************************************************/
uint32_t OS_InterLockedInc( void* pvVal)
{
#if _WIN32
  uint32_t volatile *plVar = (uint32_t volatile*)pvVal;
  /* access to 32-bit variables will be atomic on 32-bit platform (see OS_CreateInterLockedVariable()) */
#else
  /* access to 64-bit variables will be atomic on 64-bit platform (see OS_CreateInterLockedVariable()) */
  uint64_t volatile *plVar = (uint64_t volatile*)pvVal;
#endif
  return (uint32_t)(++(*plVar));
}

/*****************************************************************************/
/*! Decrements the current value in an atomic fashion
*     \param pvVal Pointer to the interlocked variable allocated by
                   OS_CreateInterLockedVariable()
*     \return current value-1                                                */
/*****************************************************************************/
uint32_t OS_InterLockedDec( void* pvVal)
{
#if _WIN32
  /* access to 32-bit variables will be atomic on 32-bit platform (see OS_CreateInterLockedVariable()) */
  uint32_t volatile *plVar = (uint32_t volatile*)pvVal;
#else
  /* access to 64-bit variables will be atomic on 64-bit platform (see OS_CreateInterLockedVariable()) */
  uint64_t volatile *plVar = (uint64_t volatile*)pvVal;
#endif
  return (uint32_t)(--(*plVar));
}

/*****************************************************************************/
/*! Returns the current value in an atomic fashion
*     \param pvVal Pointer to the interlocked variable allocated by
                   OS_CreateInterLockedVariable()
*     \return current value                                                  */
/*****************************************************************************/
uint32_t OS_InterLockedRead( void* pvVal)
{
#if _WIN32
  /* access to 32-bit variables will be atomic on 32-bit platform (see OS_CreateInterLockedVariable()) */
  uint32_t volatile *plVar  = (uint32_t volatile*)pvVal;
#else
  /* access to 64-bit variables will be atomic on 64-bit platform (see OS_CreateInterLockedVariable()) */
  uint64_t volatile *plVar  = (uint64_t volatile*)pvVal;
#endif
  if (NULL != plVar)
    return (uint32_t)*plVar;
  else
    return 0;
}

/*****************************************************************************/
/*! Sets value of the current variable to the given value in an atomic fashion
*     \param pvVal Pointer to the interlocked variable allocated by
*     \return current value                                                  */
/*****************************************************************************/
uint32_t OS_InterLockedSet( void* pvVal, uint32_t ulVal)
{
#if _WIN32
  /* access to 32-bit variables will be atomic on 32-bit platform (see OS_CreateInterLockedVariable()) */
  uint32_t volatile *plVar  = (uint32_t volatile*)pvVal;
  if (plVar)
    *plVar = ulVal;
#else
  /* access to 64-bit variables will be atomic on 64-bit platform (see OS_CreateInterLockedVariable()) */
  uint64_t volatile *plVar  = (uint64_t volatile*)pvVal;
  if (plVar)
    *plVar = ulVal;
#endif
  if (NULL != plVar)
    return ulVal;
  else
    return 0;
}